"""Minimal kernel builder facade for the mass‑gap simulation.

This module provides a ``get_kernel`` function with the same signature
expected by the mass‑gap core.  In the original simulation the kernel
returned a set of gauge‑dependent matrices constructed from spectral
data.  Here we simply attempt to load the U1 kernel from the shared
``data/kernels`` directory.  The returned object is not used by the
approximate mass‑gap calculation but the function exists to satisfy the
expected API.
"""

from __future__ import annotations

import os
import numpy as np


def get_kernel(L: int, b: float, k: float, n0: float):  # noqa: D401 (docstring inherited)
    """Return a kernel array for the given lattice and pivot parameters.

    The current implementation looks for a file named ``kernel_U1.npy`` in
    the repository's ``data/kernels`` directory relative to this file.
    If it cannot be found the function returns ``None``.  Clients should
    treat the returned object as opaque.
    """
    # Determine path to the repository root by ascending two levels
    here = os.path.dirname(os.path.abspath(__file__))
    repo_root = os.path.dirname(here)
    kernel_path = os.path.join(repo_root, 'data', 'kernels', 'kernel_U1.npy')
    if os.path.exists(kernel_path):
        try:
            return np.load(kernel_path, allow_pickle=True)
        except Exception:
            return None
    return None